/*
 * Routing routines
 */

#include <stdio.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_process_fabric.h"


#define FOR_EACH_XBAR(FP,XP)				\
  {							\
    int __x;						\
    for (__x=0; __x<FP->num_xbars; ++__x) {		\
      XP = FP->xbars[__x];				\
      {
#define END_XBAR_LOOP }}}

#define FOR_EACH_NIC(FP, HP, NP)			\
  {							\
    int __h;						\
    for (__h=0; __h<FP->num_hosts; ++__h) {		\
      int __n;						\
      HP = FP->hosts[__h];				\
      for (__n=0; __n<HP->num_nics; ++__n) {		\
	NP = HP->nics[__n];				\
	{
#define END_NIC_LOOP }}}}

/*
 * local prototypes
 */
static void lf_find_min_nic_dist(struct lf_xbar *xp);
static void lf_find_max_nic_dist(struct lf_xbar *xp);
static int lf_find_clos_depth(struct lf_fabric *fp);
static int lf_propagate_clos_levels(struct lf_fabric *fp, int clos_depth);

/*
 * Label all the xbars with their "clos level".  The spine is defined to be
 * the set of all xbars sharing the minimum maximum distance to any node.
 * That is, for each xbar, find the maximum distance to any node, and call it
 * "D".  The set of xbars sharing this minimum value D is the spine. Label
 * them all as level D, then every directly connected unlabelled xbar as
 * D-1, etc.  In a clos, no xbar of level L should be connected to any xbars
 * with level other than L, L+1, or L-1.
 */
int
lf_clos_label_xbars(
  struct lf_fabric *fp)
{
  struct lf_xbar *xp;
  int *links;
  int clos_depth;
  int rc;

  rc = 0;
  links = NULL;

  /* quick trip if no xbars */
  if (fp->num_xbars == 0) return 0;

  /* zero out everyone's max distance */
  FOR_EACH_XBAR(fp, xp)
    xp->min_nic_dist = -1;
    FP_XBAR(xp)->max_nic_dist = 0;
    xp->clos_level = 0;
    FP_XBAR(xp)->bf_last_visitor = NULL;
  END_XBAR_LOOP

  /* find closest and farthest NIC to each xbar */
  FOR_EACH_XBAR(fp, xp)
    lf_find_min_nic_dist(xp);
  END_XBAR_LOOP

  /* clear last visitors, and now find the max distance to any NIC */
  FOR_EACH_XBAR(fp, xp)
    FP_XBAR(xp)->bf_last_visitor = NULL;
  END_XBAR_LOOP
  FOR_EACH_XBAR(fp, xp)
    lf_find_max_nic_dist(xp);
  END_XBAR_LOOP

  /* Find the clos depth */
  clos_depth = lf_find_clos_depth(fp);

  /* Propagate clos_depth to all xbars */
  rc = lf_propagate_clos_levels(fp, clos_depth);
  if (rc == -1) LF_ERROR(("Error propagsting clos levels"));

  if (0) {
    LF_CALLOC(links, int, clos_depth+1);

    FOR_EACH_XBAR(fp, xp)
      int l;
      int p;

      l = xp->clos_level;

      /* validate l, as this is just informational */
      if (l > clos_depth || l < 0) {
	LF_FREE(links);
	LF_ERROR(("Bad clos-level on xbar, not a clos?"));
	continue;
      }

      /* check every xbar neighbor */
      for (p=0; p<xp->num_ports; ++p) {
	struct lf_xbar *xp2;

	xp2 = LF_XBAR(xp->topo_ports[p]);
	if (xp2 != NULL
	    && (xp2->ln_type == LF_NODE_NIC
	        || (xp2->ln_type == LF_NODE_XBAR
		    && xp2->clos_level < l))) {
	  ++links[l];
	}
      }
    END_XBAR_LOOP
  }

 except:
  LF_FREE(links);
  return rc;
}

/*
 * Find the distance to the closest NIC from this xbar
 */
static void
lf_find_min_nic_dist(
  struct lf_xbar *xbar)
{
  int dist;
  struct lf_xbar *xp;
  struct lf_xbar *work_list;

  dist = 1;

  /* initialize work_list to just be first xbar */
  work_list = xbar;
  FP_XBAR(xbar)->bf_next = NULL;
  FP_XBAR(xbar)->bf_last_visitor = xbar;

  /* Breadth-first until nothing more to do */
  while (work_list != NULL) {

    /* save this list and start building new work_list */
    xp = work_list;
    work_list = NULL;

    /* process every xbar in old work_list */
    while (xp != NULL) {
      int p;

      /* Check all ports for NICs and unvisited xbars */
      for (p=0; p<xp->num_ports; ++p) {
	union lf_node *np2;

	np2 = xp->topo_ports[p];
	if (np2 == NULL) continue;

	if (np2->ln_type == LF_NODE_NIC) {
	  xbar->min_nic_dist = dist;
	  return;

	} else if (np2->ln_type == LF_NODE_XBAR
		   && FP_XBAR_N(np2)->bf_last_visitor != xbar) {

	  /* add to work_list */
	  FP_XBAR_N(np2)->bf_next = work_list;
	  work_list = LF_XBAR(np2);
	  
	  /* mark as visited to prevent multiple visitors */
	  FP_XBAR_N(np2)->bf_last_visitor = xbar;
	}
      }

      /* next in list */
      xp = FP_XBAR(xp)->bf_next;
    }

    /* increment distance for each go-round */
    ++dist;
  }
}

/*
 * Find the longest minimum distance from this xbar to any NIC by using bfs
 * until there are no more xbars.  
 */
static void
lf_find_max_nic_dist(
  struct lf_xbar *xbar)
{
  int dist;
  struct lf_xbar *xp;
  struct lf_xbar *work_list;

  dist = 1;

  /* initialize work_list to just be first xbar */
  work_list = xbar;
  FP_XBAR(xbar)->bf_next = NULL;
  FP_XBAR(xbar)->bf_last_visitor = xbar;

  /* Breadth-first until nothing more to do */
  while (work_list != NULL) {

    /* save this list and start building new work_list */
    xp = work_list;
    work_list = NULL;

    /* process every xbar in old work_list */
    while (xp != NULL) {
      int p;

      /* Check all ports for NICs and unvisited xbars */
      for (p=0; p<xp->num_ports; ++p) {
	union lf_node *np2;

	np2 = xp->topo_ports[p];
	if (np2 == NULL) continue;

	if (np2->ln_type == LF_NODE_NIC) {
	  if (dist > FP_XBAR(xbar)->max_nic_dist) {
	    FP_XBAR(xbar)->max_nic_dist = dist;
	  }

	} else if (np2->ln_type == LF_NODE_XBAR
		   && FP_XBAR_N(np2)->bf_last_visitor != xbar) {
	  struct lf_xbar *xp2;

	  xp2 = LF_XBAR(np2);

	  /* add to new work_list */
	  FP_XBAR(xp2)->bf_next = work_list;
	  work_list = xp2;
	  
	  /* mark as visited to prevent multiple visitors */
	  FP_XBAR(xp2)->bf_last_visitor = xbar;
	}
      }

      /* next in list */
      xp = FP_XBAR(xp)->bf_next;
    }

    /* increment distance for each go-round */
    ++dist;
  }
}

/*
 * propagate clos levels.  start with all xbars whose "max_nic_dist" is
 * equal to the fabric clos depth.  Set their clos_level to be clos_depth.
 * Then, propagate out to all other xbars.  
 * We may go down then back up to tag "false spines", so we need to be
 * willing to go both "up" and "down" - if "max_distance" is greater than
 * ours, subtract one from our clos level, if greater, add one.
 * If equal, print an error because I am not sure that's possible...
 */
static int
lf_propagate_clos_levels(
  struct lf_fabric *fp,
  int clos_depth)
{
  struct lf_xbar *work_list;
  int x;

  /* initialize work_list to be all xbars with max_nic_dist == clos_depth */
  work_list = NULL;
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;

    xp = fp->xbars[x];
    if (FP_XBAR(xp)->max_nic_dist == clos_depth) {
      xp->clos_level = clos_depth;

      /* tag this as visited and add to work list */
      FP_XBAR(xp)->bf_last_visitor = fp;
      FP_XBAR(xp)->bf_next = work_list;
      work_list = xp;
    }
  }

  /* Breadth-first until nothing more to do */
  while (work_list != NULL) {
    struct lf_xbar *xp;

    /* save this list and start building new work_list */
    xp = work_list;
    work_list = NULL;

      /* process every xbar in old work_list */
    while (xp != NULL) {
      int p;

      /* Check all ports for unvisited xbars */
      for (p=0; p<xp->num_ports; ++p) {
	struct lf_xbar *xp2;

	xp2 = LF_XBAR(xp->topo_ports[p]);

	/* if non-existant, non-xbar, or visited, skip it */
	if (xp2 == NULL
	    || xp2->ln_type != LF_NODE_XBAR
	    || FP_XBAR(xp2)->bf_last_visitor == fp) {
	  continue;
	}

	/* set neighbors' clos_level based on mine */
	if (FP_XBAR(xp2)->max_nic_dist < FP_XBAR(xp)->max_nic_dist) {
	  xp2->clos_level = xp->clos_level + 1;
	} else {
	  xp2->clos_level = xp->clos_level - 1;
	}

	/* add to new work_list */
	FP_XBAR(xp2)->bf_next = work_list;
	work_list = xp2;
	
	/* mark as visited to prevent multiple visitors */
	FP_XBAR(xp2)->bf_last_visitor = fp;
      }

      /* next in list */
      xp = FP_XBAR(xp)->bf_next;
    }
  }

  /* We could cleanup a little here because in a "flat-top" clos,
   * there will end up being no "level 1" nodes.  We could find the 
   * "minimum" clos level anywhere and subtract from everything
   * to make the minimum 1, but it's not necessary yet.
   */
  return 0;
}

/*
 * Find the clos depth by starting from some host, and traveling to xbars
 * with decreasing "max distance", choosing each time the largest max distance
 * which is less than our own until we can go no farther.  The max_nic_distance
 * of that xbar is the depth of the clos.  This allows us to avoid finding 
 * "false spines" which are sub-spines within the overall fabric.
 */
static int
lf_find_clos_depth(
  struct lf_fabric *fp)
{
  struct lf_nic *nicp;
  struct lf_xbar *xp;

  /* just take the first connected NIC in the fabric.
   * If really a clos, any will do.
   */
  {
    int h;

    nicp = NULL;
    xp = NULL;
    for (h=0; h<fp->num_hosts; ++h) {
      if (fp->hosts[h]->num_nics > 0) {
	nicp = fp->hosts[h]->nics[0];
	xp = LF_XBAR(nicp->topo_ports[0]);
	if (xp != NULL && xp->ln_type == LF_NODE_XBAR) {
	  break;
	}
      }
    }
    if (xp == NULL) return 0;
  }

  while (1) {
    struct lf_xbar *nxp;
    int max_max;
    int p;

    nxp = NULL;
    max_max = -1;
    for (p=0; p<xp->num_ports; ++p) {
      union lf_node *onp;

      onp = xp->topo_ports[p];
      if (onp != NULL
	  && onp->ln_type == LF_NODE_XBAR
	  && FP_XBAR_N(onp)->max_nic_dist < FP_XBAR(xp)->max_nic_dist
	  && FP_XBAR_N(onp)->max_nic_dist > max_max) {
	max_max = FP_XBAR_N(onp)->max_nic_dist;
	nxp = LF_XBAR(onp);
      }
    }

    /*
     * If no higher nodes found, then this is a spine xbar!
     */
    if (max_max == -1) {
      return FP_XBAR(xp)->max_nic_dist;

    } else {
      xp = nxp;		/* otherwise, move on to the new xbar */
    }
  }
}
